using System;
using System.Xml;

using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages
{
	/// <summary>
	/// This class is a base for all VistALink protocol request messages.
	/// </summary>
	public abstract class BaseRequestMessage : VistALinkMessage
	{
		// Constants used in XML message serialization/deserialization
		private const string XMLCONSTS_REQUEST_NODE_NAME = "Request";
		private const string XMLCONSTS_TYPE_REQUEST_ATTRIBUTE_NAME = "type";

		private readonly string _requestType;

		/// <summary>
		/// Contstructor specifying minimal required VistALink 
		/// base request message parameters. 
		/// </summary>
		/// <param name="messageType">VistALink message type.</param>
		/// <param name="sourceXSD">VistALink message XSD name.</param>
		/// <param name="requestType">VistALink message request type.</param>
		protected BaseRequestMessage( string messageType, string sourceXSD, string requestType ) 
			: base( messageType, sourceXSD )
		{
			if( requestType == null )
				throw( new ArgumentNullException( "requestType" ) );

			_requestType = requestType;
		}

		/// <summary>
		/// Constructor extending standard VistALink message deserialization constructor. 
		/// Allows recreating message object from XML document.
		/// </summary>
		/// <param name="messageType">VistALink message type.</param>
		/// <param name="sourceXSD">VistALink message XSD name.</param>
		/// <param name="requestType">VistALink message request type.</param>
		/// <param name="sourceDoc">Source XML document.</param>
		protected BaseRequestMessage( string messageType, string sourceXSD, string requestType, XmlDocument sourceDoc ) 
			: base( messageType, sourceXSD, sourceDoc )
		{
			if( requestType == null )
				throw( new ArgumentNullException( "requestType" ) );

			XmlElement _requestElement = XmlParseGetRequestElement( sourceDoc );

			XmlUtility.ParseCheckRequiredAttributeValue( _requestElement, XMLCONSTS_TYPE_REQUEST_ATTRIBUTE_NAME, requestType );

			_requestType = requestType;
		}

		/// <summary>
		/// XML parsing utility method. Returns reference to request element in 
		/// the specified XML document containing XML representation of VistALink 
		/// base request message.
		/// 
		/// If element not found or more than one instance was found - 
		/// XmlParseException exception will be thrown.
		/// </summary>
		/// <param name="sourceDoc">Source XML document</param>
		/// <returns>Reference to request element.</returns>
		protected static XmlElement XmlParseGetRequestElement( XmlDocument sourceDoc )
		{
			return XmlUtility.ParseGetRequiredElementByUniqueTagName( sourceDoc, XMLCONSTS_REQUEST_NODE_NAME );
		}

		/// <summary>
		/// XML parsing method used during message recognition and deserialization.
		/// Returns request type string contained in message XML.
		/// Source XML document must have a request element, otherwise method will throw parsing exception.
		/// </summary>
		/// <param name="sourceDoc">Source XmlDocument to analyze.</param>
		/// <returns>Parsed request type contained in supplied XML.</returns>
		public static string XmlParseGetRequestTypeString( XmlDocument sourceDoc )
		{
			if( sourceDoc == null )
				throw( new ArgumentNullException( "sourceDoc" ) );

			return XmlParseGetRequestElement( sourceDoc ).GetAttribute( XMLCONSTS_TYPE_REQUEST_ATTRIBUTE_NAME );
		}

		/// <summary>
		/// Standard VistALink message XML serialization method, 
		/// translating message object into XML representation.
		/// </summary>
		/// <param name="writer">XmlWriter object to use</param>
		protected override void WriteXmlMessageBody( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.WriteStartElement( XMLCONSTS_REQUEST_NODE_NAME );
			writer.WriteAttributeString( XMLCONSTS_TYPE_REQUEST_ATTRIBUTE_NAME, _requestType );

			WriteXmlRequestAttributes( writer );

			WriteXmlRequestNodeContent( writer );

			writer.WriteEndElement();
		}

		/// <summary>
		/// Derived classes should override this method to add attributes to request element 
		/// of XML message representation during XML serialization. 
		/// </summary>
		/// <param name="writer">XmlWriter to use</param>
		protected virtual void WriteXmlRequestAttributes( XmlWriter writer ){}

		/// <summary>
		/// Derived classes should override this method to add content to request element 
		/// of XML message representation during XML serialization. 
		/// </summary>
		/// <param name="writer">XmlWriter to use</param>
		protected virtual void WriteXmlRequestNodeContent( XmlWriter writer ){}
	}
}
